/** @file   MPathFinderAlgo.h
 * @brief   Declaration of Path finding interfaces and base classes.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/02/24 12:22:00 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_MPATHFINDERALGO_H
#define H_ENG2D_MPATHFINDERALGO_H

#include <vector>
#include "Vec2D.h"

namespace eng2d {

/** @class  MPathNode
 * @brief   Baseclass for nodes that construct the path.
 * @author  Tomi Lamminsaari
 */
class PathNode
{
public:
  /** Constructor.
   */
  PathNode();
  
  /** Constructor.
   * @param   aPos              Location of this node.
   */
  PathNode( const Vec2D& aPos );
  
  /** Destructor.
   */
  virtual ~PathNode();

  /** Smaller than operator.
   */
  bool operator < ( const PathNode& aAnother ) const;
  
  /** Greater than operator.
   */
  bool operator > ( const PathNode& aAnother) const;
  
public:
  Vec2D iPosition;
  Vec2D iParentPosition;
  float iGone;
  float iHeuristic;
  float iTotalCost;
};



/** @class  MMapDataProvider
 * @brief   Declares the interface for object that provides map data to the
 *          path algorithm.
 * @author  Tomi Lamminsaari
 */
class MMapDataProvider
{
public:
  /** Destructor.
   */
  virtual ~MMapDataProvider() { }
  
  /** Approximates the distance between the given points. The closer they
   * they are, the smaller the returned value must be. Used by the path
   * finding algorithms.
   * @param   aPoint1           From this point
   * @param   aPoint2           To this point.
   * @return  Distance approximation between the points.
   */
  virtual float ApproximateDistance( const Vec2D& aPoint1,
                                     const Vec2D& aPoint2 ) const = 0;
  
  /** Tells if the two points are so close to each other that practically they
   * are the same point. This method is used by the path finding algorithms
   * to determine if two nodes represent the same position in the map.
   * @param   aPoint            The point
   * @param   aGoal             The goal coordinate.
   * @return  True, if the two points are the same node/cell in practice.
   */
  virtual bool IsSameNode( const Vec2D& aPoint, const Vec2D& aGoal ) const = 0;
  
  /** This method must create the adjacent nodes around the given node. The
   * created nodes MUST reside inside the map area. The constructed array
   * MUST NOT contain nodes that are located on an obstackle.
   * @param   aAdjacentNodes    An array for coordinates that construct the
   *                            adjacent "cells" where the route could go.
   * @param   aCurrentPos       Current node position.
   */
  virtual void GetAdjacentNodes( std::vector<Vec2D>& aAdjacentNodes,
                                 const Vec2D& aCurrentPos ) const = 0;
                                 
  /** Returns the terrain factor. The bigger the values are, the more difficult
   * the terrain will be on given coordinate.
   * @param   aPosition         Coordinate.
   * @return  Terrain goodness value for given position.
   */
  virtual float GetTerrainFactor( const Vec2D& aPosition ) const = 0;
};



/** @class  MPathFinderAlgo
 * @brief   An interface to object that provides path finding service.
 * @author  Tomi Lamminsaari
 *
 */
class MPathFinderAlgo
{
public:

  /** Datatype for path node array. */
  typedef std::vector<Vec2D> NodeArray;
  

  /** Destructor
   */
  virtual ~MPathFinderAlgo() { }

  ///
  /// New public methods
  /// ==================

  /** Finds the path from node to another. This method must construct a route
   * from point to another. This object stores the route and it can be queried
   * with GetPathNodes() method.
   * @param   aStart            From this point.
   * @param   aGoal             To this point.
   * @return  Did we find a route from point to another.
   */
  virtual bool FindPath( const Vec2D& aStart, const Vec2D& aGoal ) = 0;
                         
  /** Returns the array of the path nodes. This method is guaranteed to return
   * a valid path only if FindPath() method found a route.
   * @return  Reference to the vector containing the route.
   */
  virtual const NodeArray& GetPathNodes() const = 0;
  
  /** Erases the previous calculated route that is maintained inside this
   * object.
   */
  virtual void ResetPath() = 0;
};

};  // end of namespace

#endif
